function loadedData = load_cont_data(dirRoot, dirSubject, session, refChannels, norm_z, remove_emg, remove_type)
% Basic dataloader to load continuous data (it must be split afterwards)
% Outputs:
%     - loadedData: Fieldtrip struct containing the continuous data
%     
% Inputs:
%     - dirRoot: string with the main directory containing the data
%     - dirSubject: struct of the directory of the specific subject
%     - session: whether get objects (session = 1) or scenes (session = 2)
%     - refChannels: indicate reference of channels (none, all, specific channels)
%     - norm_z: z score the data
%     - remove_emg: boolean to remove the EMG artifacts detected manually
%     - remove_type: removal method for the EMG artifacts

    %% Load both retrieval tasks from the session
    tmp          = load(fullfile(dirRoot,dirSubject(session).name,'ret1',strcat(dirSubject(session).name,'_ret1.mat')));
    data_r1       = tmp.ret1;

    tmp          = load(fullfile(dirRoot,dirSubject(session).name,'ret2',strcat(dirSubject(session).name,'_ret2.mat')));
    data_r2       = tmp.ret2;

    %% Get EEG data
    cfg                    = [];
    cfg.channel            = {'EEG1010', '-M1', '-M2'}; 
    cfg.removemean         = 'no';
    cfg.bpfilter   = 'yes';
    cfg.bpfreq     = [0.1 45];
    cfg.bpinstabilityfix = 'reduce';

    data_eeg1            = ft_preprocessing(cfg, data_r1);
    data_eeg2            = ft_preprocessing(cfg, data_r2);                            

    cfg            = [];
    cfg.reref      = 'yes';
    cfg.refmethod  = 'avg';
    cfg.refchannel = refChannels;
    
    % Reref eeg if indicated
    if ~isempty(refChannels)
        data_eeg1            = ft_preprocessing(cfg, data_eeg1);
        data_eeg2            = ft_preprocessing(cfg, data_eeg2);
    end
        
    %% Get respiration data
    cfg               = [];
    cfg.channel       = {'Resp1' 'Resp2'};
    cfg.reref         = 'yes';
    cfg.refchannel    = {'Resp2'}; 
    ret1_resp         = ft_preprocessing(cfg,data_r1);
    
    % only keep one channel, and rename the resp channel
    cfg              = [];
    cfg.channel      = 'Resp1';
    ret1_resp        = ft_selectdata(cfg, ret1_resp);
    ret1_resp.label  = {'resp'};
    
    cfg               = [];
    cfg.channel       = {'Resp1' 'Resp2'};
    cfg.reref         = 'yes';
    cfg.refchannel    = {'Resp2'}; 
    ret2_resp         = ft_preprocessing(cfg,data_r2);
    
    % only keep one channel, and rename the resp channel
    cfg              = [];
    cfg.channel      = 'Resp1';
    ret2_resp        = ft_selectdata(cfg, ret2_resp);
    ret2_resp.label  = {'resp'}; 
    
    % Very conservative filter for respiration
    cfg                    = [];
    cfg.channel            = {'resp'}; 
    cfg.removemean         = 'no';
    cfg.bpfilter   = 'yes';
    cfg.bpfreq     = [0.1 20];
    cfg.bpinstabilityfix = 'reduce';

    ret1_resp            = ft_preprocessing(cfg, ret1_resp);
    ret2_resp            = ft_preprocessing(cfg, ret2_resp);
    
    %% Append both eeg and resp for each retrieval separately
    cfg = [];

    data_r1 = ft_appenddata(cfg, data_eeg1, ret1_resp);
    data_r2 = ft_appenddata(cfg, data_eeg2, ret2_resp); 
    
    %% Remove EMG artifact for both retrievals
    if remove_emg
        cfg = [];
        cfg.artfctdef.reject = remove_type;

        artifs = readmatrix(fullfile(dirSubject(session).folder, dirSubject(session).name,"emg_artifact_trials_r1.txt"));
        cfg.artfctdef.jump.artifact = artifs;
        data_r1 = ft_rejectartifact(cfg, data_r1);

        artifs = readmatrix(fullfile(dirSubject(session).folder, dirSubject(session).name,"emg_artifact_trials_r2.txt"));
        cfg.artfctdef.jump.artifact = artifs;
        data_r2 = ft_rejectartifact(cfg, data_r2);
    end
       
    %% zscore data
    if norm_z
        data_r1_trials = zscore([data_r1.trial{:}], [], 2);
        data_r2_trials = zscore([data_r2.trial{:}], [], 2);

        data_r1.trial{1,1} = data_r1_trials;
        data_r2.trial{1,1} = data_r2_trials;
    end
    
    %% Append retrievals
    data_all = ft_appenddata([], data_r1, data_r2);
    
    %% Output
    loadedData = data_all;
end