function [data_split, timing_trials] = splitTrials_loc(data_loc, feat, timeRange, window_radii, norm, dirSubject, session)
% Basic dataloader to load the phase of respiration per retrieval trial:
% Outputs:
%     - data_split: Fieldtrip struct with the split of localizer data
%     - timing_trials: time distance to cue for each trial (in case that
%     feat is not "cue")
%     
% Inputs:
%     - data_loc: Fieldtrip struct containing the continuous localizer data

%% Force the peaks/troughs to be contained in a trial
timing_trials = [];

% start respiration detection       
idx_resp   = find(strcmp(data_loc.label, 'resp'));
resp_sleep = data_loc.trial{1, 1}(idx_resp,:);
srate      = data_loc.fsample;

% [b, a] = butter(2, 0.5/(srate/2), 'low');
% resp_sleep = filtfilt(b, a, resp_sleep);

resp_sleep(isoutlier(resp_sleep)) = 0;

dataType   = 'humanAirflow';
bmObj      = breathmetrics(resp_sleep, srate, dataType);

% verbose prints out steps as they go. Set to 0 to silence.
verbose=0; 

% The sliding window is the most accurate method but takes slightly longer
baselineCorrectionMethod = 'sliding'; 

zScore=1;

bmObj.correctRespirationToBaseline(baselineCorrectionMethod, zScore, verbose)
bmObj.estimateAllFeatures;
breathFeat = bmObj.(feat); % inhalePeaks // exhaleTroughs

% Get triggers and metadata paths
trigFile = dir(fullfile(dirSubject(session).folder, dirSubject(session).name, 'loc','*.vmrk'));
trigFile = fullfile(trigFile.folder, trigFile.name);

trialData = dir(fullfile(dirSubject(session).folder, dirSubject(session).name, 'loc', '*.txt'));
trialData = fullfile(trialData.folder, trialData.name);

original_fs = 1000;
trial_trig = 'S 12';

% Read trials data
trig = ft_read_event(trigFile);

% down-sample triggers    
for itrig = 1:numel(trig)
    trig(itrig).sample = round(trig(itrig).sample / original_fs * data_loc.fsample);
end

% Get only trial onset triggers
trigMask = arrayfun(@(x) strcmp(x.value, trial_trig), trig);
trig = trig(trigMask);
trigSamples = [trig.sample];

% Get inhalation peaks in trials
inhTrialsMask = arrayfun(@(x) any(((x - trigSamples) < timeRange(2)*srate).*((x - trigSamples) > timeRange(1)*srate)), breathFeat);
inhTrials = breathFeat(inhTrialsMask);                                  

% Now do the opposite for trial information
inhTrialsMask = arrayfun(@(x) any(((breathFeat - x) < timeRange(2)*srate).*((breathFeat - x) > timeRange(1)*srate)), trigSamples);
trialOnsets = trigSamples(inhTrialsMask);

[SubjectNumber RunNumber	TrialNumber	StimType ImageName	Presentation Response RT	Acc	BaselineDur	ExperimentTime] = ...
    textread(trialData,...
    '%d %d %d %d %s %d %s %f %d %f %s','headerlines',1,'delimiter','\t');

for ientry=1:numel(SubjectNumber)      
    trialinfo{ientry,1}.SubjectNumber   = SubjectNumber(ientry);
    trialinfo{ientry,1}.RunNumber             = RunNumber(ientry);
    trialinfo{ientry,1}.TrialNumber     = TrialNumber(ientry);
    trialinfo{ientry,1}.Stimtype            = StimType(ientry);
    trialinfo{ientry,1}.ImageName           = ImageName(ientry);
    trialinfo{ientry,1}.Presentation         = Presentation(ientry);
    trialinfo{ientry,1}.Response         = Response(ientry);
    trialinfo{ientry,1}.RT = RT(ientry);
    trialinfo{ientry,1}.Acc             = Acc(ientry);
    trialinfo{ientry,1}.BaselineDur= BaselineDur(ientry);
    % trialinfo{ientry,1}.ExperimentTime= ExperimentTime(ientry);
end

% Remove first peak/trough from trials with 2 peaks/troughs
if(length(inhTrials)>length(trialOnsets))
    trialOnsetMask = arrayfun(@(x) sum(((inhTrials - x) < timeRange(2)*srate).*((inhTrials - x) > timeRange(1)*srate)), trialOnsets);
    while(length(inhTrials)>length(trialOnsets))
        removeIdx = find(trialOnsetMask > 1, 1, 'first');                
        inhTrials(removeIdx:removeIdx+trialOnsetMask(removeIdx)-2) = [];
        trialOnsetMask(removeIdx) = 1;
    end
end

% Split trials according to the inhalation peaks
trl = [];
time_pretrigger = window_radii;
time_posttrigger = window_radii;

pretrig = round(time_pretrigger*srate);
posttrig = round(time_posttrigger*srate);

% inhTrials = inhTrials + srate; % To go for flows

for j  = 1:numel(inhTrials)
    offset   = -pretrig;
    trlbegin = inhTrials(j) - pretrig;
    trlend   = inhTrials(j) + posttrig;
    newtrl   = [trlbegin trlend offset];
    trl = [trl; newtrl];
end

% Remove incomplete trials
tooEarly = trl(:,1)<data_loc.sampleinfo(1);
tooLate  = trl(:,2)>data_loc.sampleinfo(2);
trl(or(tooEarly, tooLate),:) = [];

inhTrialsIdx = find(inhTrialsMask);
inhTrialsIdx(or(tooEarly, tooLate)) = [];

inhTrials(or(tooEarly, tooLate)) = [];
trialOnsets(or(tooEarly, tooLate)) = [];

% Do new split
cfg                 = [];
cfg.trl             = trl;
cfg.allowoverlap    = 0;
data_split         = ft_redefinetrial(cfg, data_loc);

data_split.trialinfo = trialinfo(inhTrialsIdx);

% zscore data
if norm
    trial_data = [data_split.trial{:}]';
    preprocess_param   = mv_get_preprocess_param('zscore');
    [~, trial_data ]      = mv_preprocess_zscore(preprocess_param, trial_data);        
    all_trials     = trial_data';

%                 all_trials = zscore([data_split.trial{:}], [], 2);
    all_trials = reshape(all_trials', [], length(inhTrialsIdx), size(all_trials,1));
    for itrial = 1:size(all_trials,2)
        data_split.trial{1,itrial} = squeeze(all_trials(:,itrial,:))';
    end
end

timing_trials = [timing_trials, (inhTrials - trialOnsets)/srate];

assert(length(inhTrials) == length(inhTrialsIdx))
