function [MI_matrix_raw] = orig_PAC_resp(cfg,data_phase, data)

%% Check if Fieldtrip is in the MATLAB path
try
    ft_defaults;
catch
    error('Add Fieldtrip to your MATLAB path');
end

% Get sampling frequency
Fs = ft_getopt(cfg,'Fs',[]);

if isempty(Fs)
    error('Please Specify cfg.Fs');
elseif ~isnumeric(Fs)
    error('cfg.Fs needs to be numeric (e.g. 1000)');
end

% Get phase frequencies
phase_freqs = ft_getopt(cfg,'phase_freqs',[]);

if isempty(phase_freqs)
    error('Please Specify cfg.phase_freqs');
end

% Get amplitude frequencies
amp_freqs = ft_getopt(cfg,'amp_freqs',[max(phase_freqs):2:Fs/2]);

% Get filter order
filt_order = ft_getopt(cfg,'filt_order',4);

% Get amplitude bandwidth method
amp_bandw_method = ft_getopt(cfg,'amp_bandw_method','maxphase');
amp_bandw = ft_getopt(cfg,'amp_bandw',10);

% Get PAC Method
method = ft_getopt(cfg,'method','tort');
fprintf('Calculate PAC\n');

% Get Masking
mask = ft_getopt(cfg,'mask',[]);

% Get option for whether to average PAC over trials
avg_PAC = ft_getopt(cfg,'avg_PAC','yes');

%% Check inputs

% Check whether the inputs are numbers(!)
if ~floor(phase_freqs) == phase_freqs
    error('Numeric Values ONLY for Phase');
end

if ~floor(amp_freqs) == amp_freqs
    ft_error('Numeric Values ONLY for Amplitude');
end

% Give user a warning if using low-frequencies for phase
if min(phase_freqs) < 7 && filt_order > 3
    ft_warning(['Think about using a lower filter order '...
        '(e.g. cfg.filt_order = 3)']);
end

% If incorrect method abort and warn  the user
if ~any(strcmp({'tort','plv','ozkurt','canolty'},method))
    error(['Specified PAC method ''' method ''' not supported']);
end

% Check whether PAC can be detected
switch amp_bandw_method
    
    case 'number'
        % If the bandwidth is less than the maximum phase frequency...
        if amp_bandw < max(phase_freqs)
            
            error(['You will not be able to detect PAC with this configuration.'...
                ' Reduce the phase to ' ...
                num2str(amp_bandw) 'Hz, or increase the amplitude bandwidth to '...
                num2str(max(phase_freqs)+1) 'Hz']);
        end
    case 'maxphase'
        % If minimum
        if min(amp_freqs) - max(phase_freqs)*1.5 < max(phase_freqs)
            error(['You will not be able to detect PAC with this configuration.'])
        end
    case 'centre_freq'
        % If
        if min(amp_freqs)/2.5 < max(phase_freqs)
            try
                low_amp = min(amp_freqs(find(amp_freqs/2.5 > max(phase_freqs))));
            catch
                low_amp = '?';
            end
            
            error(['You will not be able to detect PAC with this configuration.'...
                ' Reduce the phase to ' ...
                num2str(min(amp_freqs)/2.5) 'Hz, or increase the amplitude to '...
                num2str(low_amp) 'Hz']);
        end
end

%% Filter Phase Frequencies & take 'angle'
disp('Filtering Phase...');

if ~isempty(mask)
    phase_filtered = zeros(size(data_phase,1),length(phase_freqs),length(data_phase(:,mask(1):...
        mask(2))));
else
    phase_filtered = zeros(size(data_phase,1),length(phase_freqs),length(data_phase));
    
end

for phase = 1:length(phase_freqs)
    try
        [filt] = ft_preproc_bandpassfilter(data_phase, Fs,...
            [phase_freqs-0.05, phase_freqs+0.05],...
            filt_order, 'but', 'twopass', 'no');
    catch
        error('Could not filter ... Perhaps try a lower filter order');
    end
    
    if ~isempty(mask)
        phase_filtered(:,phase,:) = ft_preproc_hilbert(filt(:,mask(1):...
            mask(2)), 'angle');
    else
        phase_filtered(:,phase,:) = ft_preproc_hilbert(filt, 'angle');
    end
    
    clear filt
end


%% Filter Amplitude & Take 'abs'
disp('Filtering Amplitude...');

if ~isempty(mask)
    amp_filtered = zeros(size(data,1),length(amp_freqs),length(data(:,mask(1):...
        mask(2))));
else
    amp_filtered = zeros(size(data,1),length(amp_freqs),length(data));
end

for amp = 1:length(amp_freqs)
    
    % Switch based on bandwidth method
    switch amp_bandw_method
        
        case 'number'
            
            if amp_freqs(amp) == .5
                Af1 = .2;
            else
            
            Af1 = amp_freqs(amp) - amp_bandw;
            end
            
            Af2 = amp_freqs(amp) + amp_bandw;
            
        case 'maxphase'
            if amp == 1
                fprintf('Bandwidth = %.1fHz\n',1.5.*max(phase_freqs));
            end
            %
            Af1 = amp_freqs(amp) - 1.5*max(phase_freqs);
            Af2 = amp_freqs(amp) + 1.5*max(phase_freqs);
            
        case 'centre_freq'
            if amp == 1
                fprintf('Bandwidth = 2.5* centre amplitude frequency\n')
            end
            
            Af1 = round(amp_freqs(amp) -(amp_freqs(amp)/2.5));
            Af2 = round(amp_freqs(amp) +(amp_freqs(amp)/2.5));
                      
    end
    
    % Filter
    [filt] = ft_preproc_bandpassfilter(data, Fs,...
        [Af1 Af2],filt_order, 'but', 'twopass', 'no');
    
    % Take abs (and mask values if required)
    if ~isempty(mask)
        amp_filtered(:,amp,:) = ft_preproc_hilbert(filt(:,mask(1):...
            mask(2)), 'abs');
    else
        amp_filtered(:,amp,:) = ft_preproc_hilbert(filt, 'abs');
    end
    
    clear filt Af1 Af2
end

%% PAC computation
MI_matrix_raw = zeros(size(data,1),length(amp_freqs),length(phase_freqs));

for trial = 1:size(data,1)
    for phase = 1:length(phase_freqs)
        for amp = 1:length(amp_freqs)
            
            phase_data = squeeze(phase_filtered(trial,phase,:));
            amp_data = squeeze(amp_filtered(trial,amp,:));
            
            % Switch based on the method of PAC computation
            switch method
                case 'tort'
                    try
                    [MI] = calc_MI_tort(phase_data,amp_data,20);
                    catch
                        [MI] = math.calc_MI_tort(phase_data,amp_data,20);
                    end
                    
                case 'ozkurt'
                    [MI] = calc_MI_ozkurt(phase_data,amp_data);
                    
                case 'plv'
                    [MI] = cohen_PLV(phase_data,amp_data);
                    
                case 'canolty'
                    [MI] = calc_MI_canolty(phase_data,amp_data);
            end
            
            % Add to matrix outside the loop
            MI_matrix_raw(trial,amp,phase) = MI;
        end
    end
end

%% Average PAC over trials if specified

if strcmp(avg_PAC,'yes')
    MI_matrix_raw = squeeze(mean(MI_matrix_raw,1));
    
elseif strcmp(avg_PAC,'no');
    disp('Returning PAC values per trial');
else
    ft_warning('Please specify cfg.avg_PAC = ''yes'' or ''no''');
end