function data_phase = load_segm_phase(dirRoot, dirSubject, session, hilbert, remove_artifs)
% Basic dataloader to load the phase of respiration per retrieval trial:
% Outputs:
%     - data_phase: Fieldtrip struct containing the phase of respiration
%     
% Inputs:
%     - dirRoot: string with the main directory containing the data
%     - dirSubject: struct of the directory of the specific subject
%     - session: whether get objects (session = 1) or scenes (session = 2)
%     - hilbert: boolean to indicate the use of hilbert (hilbert = true) or double interpolation (hilbert = false)
%     - remove_artifs: boolean to remove the respiratory artifacts detected manually
    
% First we check whether we want to use Hilbert o Double interpolation
% methods, and we load both pre- and post-sleep retrieval tasks
    if hilbert
        tmp          = load(fullfile(dirRoot,dirSubject(session).name,'ret1',strcat(dirSubject(session).name,'_ret1_resp_segm_Hilbert.mat')));
        data_r1       = tmp.ret1_resp;

        tmp          = load(fullfile(dirRoot,dirSubject(session).name,'ret2',strcat(dirSubject(session).name,'_ret2_resp_segm_Hilbert.mat')));
        data_r2       = tmp.ret2_resp;

    else
        tmp          = load(fullfile(dirRoot,dirSubject(session).name,'ret1',strcat(dirSubject(session).name,'_ret1_resp_segm_DoubleInterp.mat')));
        data_r1       = tmp.ret1_resp;

        tmp          = load(fullfile(dirRoot,dirSubject(session).name,'ret2',strcat(dirSubject(session).name,'_ret2_resp_segm_DoubleInterp.mat')));
        data_r2       = tmp.ret2_resp;
    end

    % bring data into better format
    cfg                    = [];
    cfg.channel            = {'resp'}; 
    cfg.removemean         = 'no';

    data_r1            = ft_preprocessing(cfg, data_r1);
    data_r2            = ft_preprocessing(cfg, data_r2);                            

    cfg                    = [];
    cfg.keeptrials         = 'yes';
    cfg.removemean         = 'no';

    data_r1            = ft_timelockanalysis(cfg, data_r1);
    data_r2            = ft_timelockanalysis(cfg, data_r2);

    % add conditions (session 1: objects, session 2: scenes)
    for i = 1:size (data_r1.trialinfo,1)
            data_r1.trialinfo{i, 1}.Stimtype = session;
    end

    for i = 1:size (data_r2.trialinfo,1)
            data_r2.trialinfo{i, 1}.Stimtype = session;
    end     

    % Append both pre- and post-sleep together
    trialinfo_all = [data_r1.trialinfo; data_r2.trialinfo];
    data_all = ft_appendtimelock([], data_r1, data_r2);
    data_all.trialinfo = trialinfo_all;

    % Remove artifacts (if marked file exists)
    if remove_artifs && isfile(fullfile(dirSubject(session).folder, dirSubject(session).name, "artifact_trials.txt"))
        artifs = readmatrix(fullfile(dirSubject(session).folder, dirSubject(session).name,"artifact_trials.txt"));
        
        deletedTrials_start = ceil(artifs(:,1)/length(data_all.time));
        deletedTrials_end = ceil(artifs(:,2)/length(data_all.time));
        deletedTrials = unique([deletedTrials_start; deletedTrials_end]);
        
        % class selection (2 classes)
        trlinfo  = cell2mat(data_all.trialinfo);   
        accuracy = [trlinfo.ExemplarAccuracy];   % 1 [correct], 0 [wrong] 
        oldnew   = [trlinfo.OldNew];

        rem_idx      = accuracy==1 & oldnew == 1;
        nrem_idx      = accuracy==0 & oldnew == 1;
    
        % This is to keep track of deleted trials, it may be needed for
        % posterior processings
        rem_deleted_idx = find(rem_idx(deletedTrials));
        cum_rem_idx = cumsum(rem_idx);
        deletedTrials_rem = cum_rem_idx(deletedTrials(rem_deleted_idx));
        
        nrem_deleted_idx = find(nrem_idx(deletedTrials));
        cum_nrem_idx = cumsum(nrem_idx);
        deletedTrials_nrem = cum_nrem_idx(deletedTrials(nrem_deleted_idx));
        
        % Reject the artifacts
        cfg = [];
        cfg.artfctdef.visual.artifact = artifs;
        data_all = ft_rejectartifact(cfg, data_all);
        
        % Save in the legacy cfg for later
        data_all.cfg.deletedArtifs = deletedTrials;
        data_all.cfg.deletedTrials_rem = deletedTrials_rem;
        data_all.cfg.deletedTrials_nrem = deletedTrials_nrem;
    end
    
    data_phase = data_all;
end